/*
	SdkLayer:

	This object is the abstract facade that 3rd parties communicate with to control a layer, as returned
	from a behavior layer param.
	
*/

define(["src/utils", "lodash", "src/math/Vec2"],
function(utils, lodash, vec2) {
	"use strict";


	function SdkLayer(inLayer) {
		this.privateLayer = inLayer;	// privateLayer might be a Layer or a TrackItem
		// to make this truly private from 3rd parties, we'd have to use a closure here for every entry point,
		//	which could be slow and cumbersome. And having the option to cheat could be helpful for 3rd parties,
		//	as long as they know they're cheating and need to request new APIs.
		
		this.sharedBehaviorFrameData = {};	// cleared out after each frame (@@@ clear out in engine)
		this.sharedBehaviorSceneData = {};	// persists across frames
	}
	
	utils.mixin(SdkLayer, {
		getSource :		function ()			{ return this.privateLayer.getSource(); },
		getName :		function ()			{ return this.privateLayer.getName(); },
		getId :			function ()			{ return this.privateLayer.getStageId(); }, // only stable within a single stage execution
		getPuppet :		function ()			{ return this.privateLayer.getPuppet(); },	// same as getSource if source is a puppet; TODO: wrap to return SdkPuppet
		getParentPuppet : function ()		{ return this.privateLayer.parentPuppet; },	// returns null when this layer is TrackItem
		getParentLayer : function ()		{ var p = this.getParentPuppet();		// returns null when this layer is TrackItem
											 	if (p) {
													return p.getParentLayer().getSdkLayer();
												} else {
													return null;
												}
											},

		// for inter-behavior communication; cleared at end of each frame automatically
		//	TODO: move implemention & storage into privateLayer once TrackItems are sourced by Layers (so that stageLayer will have this data)
		setSharedFrameData : function (key, value) { this.sharedBehaviorFrameData[key] = value; },
		getSharedFrameData : function (key)	{ return this.sharedBehaviorFrameData[key]; },
		
		// this one should _only_ be exposed on the privateLayer
		clearSharedFrameData : function () { this.sharedBehaviorFrameData = {}; },
		
		// like the FrameData calls above, but persistent for the life the stage (until refreshed/reopened)
		setSharedSceneData : function (key, value) { this.sharedBehaviorSceneData[key] = value; },
		getSharedSceneData : function (key)	{ return this.sharedBehaviorSceneData[key]; },


		setOpacity :			function (opacity)	{ this.privateLayer.setOpacity(opacity); },
		getOpacity :			function ()			{ return this.privateLayer.getOpacity(); },
		getBindingOpacity :		function ()			{ return this.privateLayer.getBindingOpacity(); },
		setBlendMode :			function (blendMode){ return this.privateLayer.setBlendMode(blendMode); },
		getBlendMode :			function ()			{ return this.privateLayer.getBlendMode(); },
		getBindingBlendMode :	function ()			{ return this.privateLayer.getBindingBlendMode(); },
		setVisible :			function (bEnabled)	{ return this.privateLayer.setVisible(bEnabled); }, // TODO: remove; deprecated in favor of setTriggerable()/trigger()
		getVisible :			function ()			{ return this.privateLayer.getVisible(); },			// TODO: remove; deprecated in favor of setTriggerable()/trigger()
		
		setTriggerable :		function (bHideSiblings)	{ return this.privateLayer.setTriggerable(bHideSiblings); },
		//getTriggerable :		function ()					{ return this.privateLayer.getTriggerable(); }, not exposed until shown to be needed by behaviors
		trigger :				function (priority0)		{ return this.privateLayer.setTriggered(true, priority0); }, // no way to clear trigger via sdk; happens after each frame
		getTriggered :			function ()					{ return this.privateLayer.getTriggered(); },
		getTriggeredByMap :		function ()					{ return this.privateLayer.getTriggeredByMap(); },

		// may return null
		getKeyTriggerKey :				function ()		{ return this.privateLayer.getKeyTriggerKey(); },
		getHideOthersWhenTriggered :	function ()		{ return this.privateLayer.getHideOthersWhenTriggered(); },
		getTriggerUsesLatch :			function ()		{ return this.privateLayer.getTriggerUsesLatch(); },

		// may want to implement getBounds on Layerish and call it from here instead of calling getSource here?
		getBounds : 	function ()			{ 
			var layer = this.privateLayer,
				bounds = layer.getSource().getBounds(),
				originSource = vec2.of(bounds[0], bounds[1]),
				sizeSource = vec2.of(bounds[2], bounds[3]),
				matLayer_Source = layer.getSourceMatrixRelativeToLayer();

			var originLayer = vec2.transformAffine(matLayer_Source, originSource),
				sizeLayer = vec2.transformLinear(matLayer_Source, sizeSource);

			bounds[0] = originLayer[0];
			bounds[1] = originLayer[1];
			bounds[2] = sizeLayer[0];
			bounds[3] = sizeLayer[1];

			return bounds;
		},

		// calls the given function once for each layer, with a single sdkLayer as the argument
		// starts with the "this" layer and includes all children of sources; return false from fn to stop iterating,
		// or an object that specifies whether to continueB and skipChildrenB
		//	based on more general treeContainer.js version, but we can't use the entire treeContainer mixin because (1) we don't want a writable
		//	tree, and (2) we don't want a parallel tree one level off of layers (vs. items)
		forEachLayerBreadthFirst : function (fn) {
			var layer,
				queue = [ this ],
			    fnResult, continueB, includeChildrenB;
			
			function pushChild (lay2) {
				queue.push(lay2);
			}

			while (queue.length > 0) {
			    layer = queue.shift();

			    fnResult = fn(layer);
				
				if (lodash.isObject(fnResult)) {
					continueB = (fnResult.continueB !== false);	// undefined means continue
					includeChildrenB = (fnResult.includeChildrenB !== false); // undefined means we should include children
				} else {
					continueB = (fnResult !== false);
					includeChildrenB = true;
				}
			    if (!continueB) {
					break;
				}
				
				if (includeChildrenB && !layer.forEachDirectChildLayer(pushChild)) {
					break;
				}
					
			}

			return this;
		},
		
		// if the function returns false, the iteration stops and the return value is also false; iterates in front-to-back order, take arg sdkLayer
		forEachDirectChildLayer : function (fn) {
			var sourcePuppet, children, i;
			sourcePuppet = this.getPuppet(); // returns false if layer source isn't a puppet

			if (sourcePuppet) {
				children = sourcePuppet.getLayers();
				for (i = children.length-1; i >= 0 ; --i) {
					var layer = children[i];
					if (layer.getUserVisible()) {
						if (fn(layer.getSdkLayer()) === false) {
							return false;
						}
					}
				}
			}
			
			return true;
		}

	});
	
	return SdkLayer;
});